"""
Lösung für die Übung zur OSM-Datenabfrage und Visualisierung
=============================================================

Dieses Skript enthält die Lösung für die Übung im Arbeitsblatt.
"""

# =============================================================================
# OSM-Datenabfrage und Visualisierung
# =============================================================================

# Import der benötigten Bibliotheken
import plotly.express as px  # Für interaktive Karten
import geopandas as gpd     # Für die Arbeit mit Geodaten
from collections import namedtuple  # Für strukturierte Daten
from osm2geojson import overpass_call, json2geojson  # Für OSM-Datenabfragen

# Definition von Hilfsstrukturen für die Datenorganisation
Bbox = namedtuple("Bbox", ["south", "west", "north", "east"])  # Bounding Box für geografische Gebiete
Tag = namedtuple("Tag", ["key", "value"])  # OSM-Tags für die Suche

# Deklaration der EPSG-Codes für verschiedene Koordinatenreferenzsysteme (crs)
WGS84 = 4326
webMercator = 3857

def load_osm_from_overpass(bbox, tag, crs=f"epsg:{WGS84}") -> gpd.GeoDataFrame:
    """
    Lädt OSM-Daten von der Overpass API und konvertiert sie in ein GeoDataFrame.
    
    Args:
        bbox: Bounding Box für das Suchgebiet
        tag: OSM-Tag für die Suche (z.B. sport=table_tennis)
        crs: Koordinatenreferenzsystem (Standard: WGS84)
    
    Returns:
        GeoDataFrame mit den gefundenen OSM-Elementen
    """
    geojson = load_osm_from_overpass_geojson(bbox, tag)
    return gpd.GeoDataFrame.from_features(geojson, crs=crs)

def load_osm_from_overpass_geojson(bbox, tag):
    """
    Erstellt und führt eine Overpass API-Abfrage durch.
    
    Args:
        bbox: Bounding Box für das Suchgebiet
        tag: OSM-Tag für die Suche
    
    Returns:
        GeoJSON-Objekt mit den gefundenen OSM-Elementen
    """
    query = f"""
    [out:json];
    nwr["{tag.key}"="{tag.value}"]({bbox.south},{bbox.west},{bbox.north},{bbox.east});
    out body;
    >;
    out skel qt;
    """
    response = overpass_call(query)
    return json2geojson(response)

def plot_gdf_on_map(gdf, title=""):
    """
    Erstellt eine interaktive Karte mit den GeoDataFrame-Daten.
    
    Args:
        gdf: GeoDataFrame mit den zu visualisierenden Daten
        title: Titel der Karte
    
    Returns:
        Plotly Figure-Objekt mit der interaktiven Karte
    """
    fig = px.scatter_map(
        gdf,
        lat=gdf.geometry.y,
        lon=gdf.geometry.x,
        color_discrete_sequence=["fuchsia"],
        height=500,
        zoom=11,
    )
    fig.update_layout(title=title, map_style="open-street-map")
    fig.update_layout(margin={"r": 0, "l": 0, "b": 0})
    return fig

def main():
    """
    Hauptfunktion zum Testen der OSM-Datenabfrage.
    Führt zwei Tests durch:
    1. Tischtennistische in Zürich
    2. Trinkbrunnen in Rapperswil-Jona
    Gibt eine Karte aller Trinkwasserbrunnen in Rapperswil-Jona aus.
    """
    # Test 1: Tischtennistische in Zürich
    print("Test 1: Tischtennistische in Zürich")
    tag = Tag(key="sport", value="table_tennis")  # OSM-Tag für Tischtennistische
    bbox_zurich = Bbox(west=8.471668, south=47.348834, east=8.600454, north=47.434379)
    table_tennis_gdf = load_osm_from_overpass(bbox_zurich, tag)
    print(f"Gefundene Tischtennistische: {len(table_tennis_gdf)}")
    
    # Test 2: Trinkbrunnen in Rapperswil-Jona
    print("\nTest 2: Trinkbrunnen in Rapperswil-Jona")
    tag = Tag(key="amenity", value="drinking_water")  # OSM-Tag für Trinkbrunnen
    bbox_rapperswil = Bbox(west=8.8000, south=47.2000, east=8.9000, north=47.3000)
    drinking_water_gdf = load_osm_from_overpass(bbox_rapperswil, tag)
    print(f"Gefundene Trinkbrunnen: {len(drinking_water_gdf)}")

    # Lösung: Karte aller Trinkwasserbrunnen in Rapperswil-Jona
    tag = Tag(key='amenity', value='drinking_water')
    bbox_rapperswil_jona = Bbox(west=8.8000, south=47.2000, east=8.9000, north=47.3000)
    drinking_fountain_gdf = load_osm_from_overpass(bbox_rapperswil_jona, tag)
    # Konvertieren aller Geometrien zu Punkten (siehe https://gis.stackexchange.com/questions/302430/polygon-to-point-in-geopandas)
    # Um Berechnungsfehler zu minimieren, konvertieren wir zu EPSG:3857 (webMercator) und danach wieder zurück zu EPSG:4326 (WGS84)
    gdf = drinking_fountain_gdf.to_crs(epsg=webMercator)
    gdf.geometry = gdf.geometry.centroid
    gdf = gdf.to_crs(epsg=WGS84)
    fig = plot_gdf_on_map(gdf, title="Rapperswil-Jona Drinking Fountains")
    # Falls kein Fenster erscheint, die folgende Zeile auskommentieren und die Datei loesungen_map.html im Browser öffnen
    # fig.write_html("loesungen_map.html")
    fig.show()

if __name__ == "__main__":
    main()
